/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/. */

import * as EventUtils from "resource://testing-common/mail/EventUtils.sys.mjs";
import { TestUtils } from "resource://testing-common/TestUtils.sys.mjs";
import { PromiseTestUtils } from "resource://testing-common/mailnews/PromiseTestUtils.sys.mjs";

import {
  mc,
  get_about_3pane,
  wait_for_message_display_completion,
} from "resource://testing-common/mail/FolderDisplayHelpers.sys.mjs";

/**
 * Mark the selected messages as junk. This is done by pressing the J key.
 */
export function mark_selected_messages_as_junk() {
  get_about_3pane().document.getElementById("threadTree").focus();
  EventUtils.synthesizeKey("j", {}, mc);
}

/**
 * Delete all mail marked as junk in the selected folder. This is done by
 * activating the menu option from the Tools menu.
 *
 * @param {integer} aNumDeletesExpected - The number of deletes expected.
 * @param {nsIMsgFolder} folder - The folder the messages are in.
 */
export async function delete_mail_marked_as_junk(aNumDeletesExpected, folder) {
  const about3Pane = get_about_3pane();

  // Monkey patch and wrap around the deleteJunkInFolder function, mainly for
  // the case where deletes aren't expected.
  const realDeleteJunkInFolder = about3Pane.deleteJunkInFolder;
  let numMessagesDeleted = null;
  const fakeDeleteJunkInFolder = function (...args) {
    numMessagesDeleted = realDeleteJunkInFolder(...args);
    return numMessagesDeleted;
  };
  try {
    about3Pane.deleteJunkInFolder = fakeDeleteJunkInFolder;

    // If something is loading, make sure it finishes loading...
    await wait_for_message_display_completion();
    let completed;
    if (aNumDeletesExpected != 0) {
      completed = PromiseTestUtils.promiseFolderEvent(
        folder,
        "DeleteOrMoveMsgCompleted"
      );
    }

    about3Pane.goDoCommand("cmd_deleteJunk");

    if (aNumDeletesExpected != 0) {
      await completed;
    }

    // If timeout waiting for numMessagesDeleted to turn non-null,
    // this either means that deleteJunkInFolder didn't get called or that it
    // didn't return a value."

    await TestUtils.waitForCondition(
      () => numMessagesDeleted === aNumDeletesExpected,
      `Should have got ${aNumDeletesExpected} deletes, not ${numMessagesDeleted}`
    );
  } finally {
    about3Pane.deleteJunkInFolder = realDeleteJunkInFolder;
  }
}
